unit SrcParser;
interface
uses
  Classes;
type
  TParseState = (psNormal, psIdent, psKeyword, psNumber, psString, psComment );

type
  TParser = class
  private
    FOnFound  : TNotifyEvent;
    FOnReplace: TNotifyEvent;
    FState   : TParseState;
    FInput   : String;
    FSource  : String;
    FKeyWord : String;
    FSelStart,
    FSelEnd  : Integer;
    FComment : Integer;
    FPrior   : Boolean;
    FIndex   : Integer;
    FKeywords: TStrings;
    procedure SetKeywords(Value: TStrings);
  protected
    procedure Found; virtual;
    procedure Mark;
    procedure ParseChar(Ch,Next: Char; i: Integer); virtual; abstract;
    procedure PushChar(Ch: Char);
    function  IsKeyword(const aKey: String): Boolean;
  public
    constructor Create;
    destructor Destroy; override;
    procedure HighLight(const Prefix, Postfix: String);
    function  Parse(const S: String): String; virtual;
    property  State  : TParseState read FState;
    property  Input  : String read FInput write FInput;
    property  Source : String read FSource write FSource;
    property  Keywords: TStrings read FKeywords write SetKeywords;
    property  OnFound: TNotifyEvent read FOnFound write FOnFound;
    property  OnReplace: TNotifyEvent read FOnReplace write FOnReplace;
  end;

  TPascalParser = class(TParser)
  protected
    procedure ParseChar(Ch, Next: Char; i: Integer); override;
  public
    constructor Create;
  end;

  TAsmParser = class(TParser)
  protected
    procedure ParseChar(Ch, Next: Char; i: Integer); override;
  public
    constructor Create;
  end;

  TCppParser = class(TParser)
  protected
    procedure ParseChar(Ch, Next: Char; i: Integer); override;
  public
    constructor Create;
  end;




  function SourceToRtf(const aFile: String; aMono, aHeader: Boolean): String;
  function  strFileLoad(const aFile: String): String;

implementation

uses
  Windows, Graphics, SysUtils;

const
  DIGIT = ['0'..'9'];
  ALPHA = ['A'..'Z', 'a'..'z'];
  IDENT = ALPHA + DIGIT + ['_'];

  _Alpha : set of char = ['A'..'Z'];


const  
  STRDELIM = '''';

  { digits as chars }
  ZERO   = '0';  ONE  = '1';  TWO    = '2';  THREE  = '3';  FOUR  = '4';
  FIVE   = '5';  SIX  = '6';  SEVEN  = '7';  EIGHT  = '8';  NINE  = '9';
   { several important ASCII codes }
  NULL            =  #0;
  BACKSPACE       =  #8;
  TAB             =  #9;
  LF              = #10;
  CR              = #13;
  EOF_            = #26;
  ESC             = #27;
  BLANK           = #32;
  SPACE           = BLANK;


//////////////////////////////////

constructor TParser.Create;
begin
  inherited Create;
  FKeywords:=TStringList.Create;
  TStringList(FKeywords).Sorted:=True;
end;

destructor TParser.Destroy;
begin
  FKeywords.Free;
  inherited Destroy;
end;

procedure TParser.SetKeywords(Value: TStrings);
begin
  FKeywords.Assign(Value);
end;

function TParser.IsKeyword(const aKey: String): Boolean;
var
  L, H, I, C: Integer;
begin
  Result := False;
  L := 0;
  H := FKeywords.Count-1;
  while L <= H do
  begin
    I := (L + H) shr 1;
    C := CompareText(FKeywords[i],aKey);
    if C < 0  then L := I + 1 else
    begin
      H := I - 1;
      if C = 0 then
      begin
        Result:=True;
        break;
      end;
    end;
  end;
end;


procedure TParser.HighLight(const Prefix, Postfix: String);
begin
  Insert(Postfix,FSource,FSelEnd+1);
  Insert(Prefix,FSource,FSelStart);
end;

function TParser.Parse(const S: String): String;
var
  Ch,
  Next: Char;
begin
  FState := psNormal;
  FSource:= '';
  FInput := S;
  Findex := 1;
  while FIndex <= Length(FInput) do
  begin
    Ch := FInput[FIndex];
    if FIndex < Length(FInput) then Next:=FInput[FIndex+1] else Next:=#0;
    FSource := FSource + Ch;
    if FState = psNormal then
       FSelStart:=Length(FSource);
    if Assigned(FOnReplace) then
       FOnReplace(Self);
    ParseChar(Ch,Next,FIndex);
    Inc(FIndex);
  end;
  Result:=FSource;
end;

procedure TParser.PushChar(Ch: Char);
begin
  FSource:=FSource+ Ch;
  Inc(FIndex);
end;

procedure TParser.Found;
begin
  if Assigned(FOnFound) then FOnFound(Self);
end;

procedure TParser.Mark;
begin
  FPrior:=False;
end;

{ TPascalParser - Delphi & Pascal Parser }

constructor TPascalParser.Create;
const
  _Keywords : array[1..70] of string =
   ('AND','ARRAY','AS','ASM','BEGIN','CASE','CLASS','CONST','CONSTRUCTOR','DESTRUCTOR',
    'DIV','DO','DOWNTO','ELSE','END','EXCEPT','EXPORTS','FILE','FINALIZATION','FINALLY',
    'FOR','FUNCTION','GOTO','IF','IMPLEMENTATION','INHERITED','INITIALIZATION',
    'INLINE','INTERFACE','IS','LABEL','LIBRARY','MOD','NIL','NOT','OBJECT','OF','PACKED',
    'PROCEDURE','PUBLIC','PUBLISHED','PROGRAM','PRIVATE','PROTECTED','PROPERTY','RAISE',
    'RECORD','REPEAT','SET','SHL','SHR','STRING','THEN','THREADVAR','TRY',
    'TYPE','UNIT','UNTIL','USES','WHILE','WITH','VAR','XOR',
    'VIRTUAL','ABSTRACT','OVERRIDE','READ','WRITE','DEFAULT','INDEX' );
var
  i : Integer;

begin
  inherited Create;
  for i:= Low(_Keywords) to High(_Keywords) do
    FKeywords.Add(_Keywords[i]);
end;

procedure TPascalParser.ParseChar(Ch, Next: Char; i: Integer);
begin
  case FState of
    psIdent :
      if not (Ch in  ALPHA+[ZERO..NINE,'_']) then
      begin
        FSelEnd:=Length(FSource)-1;
        if IsKeyWord(FKeyWord) then
        begin
          FState := psKeyword;
          Found;
          { here we should step into the second level of parsing }
        end else
        begin
          { push identifier into list of identifier }
          Found;
        end;
        FState:=psNormal;
      end else
        FKeyword:=FKeyword+Upcase(Ch);

    psNumber :
      if not (Ch in [ZERO..NINE,'.','E','e']) then
      begin
        FSelEnd:=Length(FSource)-1;
        Found;
        FState:=psNormal;                           { push number on stack }
      end;

    psString :
      if Ch = STRDELIM then
         if Next <> STRDELIM then
         begin
           PushChar(Next);
           FSelEnd:=Length(FSource);
           Found;
           FState:=psNormal;                        { push string on stack }
         end;

    psNormal :
      case Ch of
        SPACE,NULL,TAB,CR,LF : ;                    { null characters }

        '>','<','=','[',']','+','-' : ;             { operators }

        ZERO..NINE :                                { numbers }
              begin
                FState:=psNumber;
              end;

        '#','$' :
              if Next in DIGIT then          { special numbers }
              begin
                FState:=psNumber;
                Mark;
              end;

        'A'..'Z','a'..'z', '_' :                   { identifier }
              begin
                FState:=psIdent;
                FKeyword:=Ch;
                Mark;
              end;

        STRDELIM:
              begin                                 { string }
                FState:=psString;
                Mark;
              end;

        '{' : begin                                 { (* comment }
                FState:=psComment;
                FComment:=1;
                Mark;
              end;

        '(' : if Next = '*' then                  { { comment  }
              begin
                FState:=psComment;
                FComment:=2;
                Mark;
              end;

        '/' : if Next = '/' then                  { // comment }
              begin
                FState:=psComment;
                FComment:=3;
                Mark;
              end;
      end;

    psComment :

      case FComment of
        1 : if Ch = '}' then
            begin
              FSelEnd:=Length(FSource);
              Found;
              FState:=psNormal;
            end;
        2 : if Ch = '*' then
               if Next = ')' then
               begin
                 PushChar(Next);
                 FSelEnd:=Length(FSource);
                 Found;
                 FState:=psNormal;
               end;
        3 : if Ch in [CR, LF] then
            begin
              FSelEnd:=Length(FSource)-1;
              Found;
              FState:=psNormal;
            end;
      end;
  end; { case state }
end;



constructor TAsmParser.Create;
const
  _Keywords : array[1..115] of string =
   ('aaa', 'aam', 'adc', 'aad', 'aas', 'add', 'and', 'call', 'cbw',
   'clc', 'cld', 'cli', 'cmc', 'cmp', 'cmpsb', 'cmpsw', 'cwd',
   'daa', 'das', 'dec', 'div', 'esc', 'hlt', 'idiv', 'imul', 'in', 'inc',
   'int', 'intro', 'iret', 'ja', 'jae', 'jb', 'jbe', 'jc', 'jcxz',
   'je', 'jg', 'jge', 'jl', 'jle', 'jmp', 'jna', 'jnae', 'jnb', 'jnbe',
   'jnc', 'jne', 'jng', 'jnl', 'jnle', 'jno', 'jnp', 'jns', 'jnz', 'jo',
   'jp', 'jpe', 'jpo', 'js', 'jz', 'lahf', 'lds', 'lea', 'les', 'lock',
   'lodsb', 'lodsw', 'loop', 'loope', 'loopne', 'loopnz', 'loopz', 'mov',
   'movsb', 'movsw', 'mul', 'neg', 'nop', 'not', 'or', 'out', 'pop', 'popf',
   'push', 'pushf', 'rcl', 'rcr', 'rep', 'repe', 'repne', 'repnz', 'repz',
   'ret', 'rol', 'ror', 'sahf', 'sal', 'sar', 'sbb', 'scasb', 'scasw', 'shl',
   'shr', 'stc', 'std', 'sti', 'stosb', 'stosw', 'sub', 'test', 'wait', 'xchg',
   'xlat', 'xor' );
var
  i : Integer;

begin
  inherited Create;
  for i:= Low(_Keywords) to High(_Keywords) do
    FKeywords.Add(_Keywords[i]);
end;

procedure TAsmParser.ParseChar(Ch, Next: Char; i: Integer);
begin
  case FState of
    psIdent :
      if not (Ch in IDENT ) then
      begin
        FSelEnd:=Length(FSource)-1;
        if IsKeyWord(FKeyWord) then
        begin
          FState := psKeyword;
          Found;
          { here we should step into the second level of parsing }
        end else
        begin
          { push identifier into list of identifier }
          Found;
        end;
        FState:=psNormal;
      end else
        FKeyword:=FKeyword+Upcase(Ch);

    psNumber :
      if not (Ch in [ZERO..NINE,'.','E','e']) then
      begin
        FSelEnd:=Length(FSource)-1;
        Found;
        FState:=psNormal;                           { push number on stack }
      end;

    psString :
      if Ch = '"' then
         if Next <>'"' then
         begin
           PushChar(Next);
           FSelEnd:=Length(FSource);
           Found;
           FState:=psNormal;                        { push string on stack }
         end;

    psNormal :
      case Ch of
        SPACE,NULL,TAB,CR,LF : ;                    { null characters }

        '>','<','=','[',']','+','-' : ;             { operators }

        ZERO..NINE :                                { numbers }
              begin
                FState:=psNumber;
              end;

        '#','$' :
              if Next in DIGIT  then          { special numbers }
              begin
                FState:=psNumber;
                Mark;
              end;

        'A'..'Z', 'a'..'z', '_' :                   { identifier }
              begin
                FState:=psIdent;
                FKeyword:=Ch;
                Mark;
              end;

        '"' : begin                                 { string }
                FState:=psString;
                Mark;
              end;

        ';' : begin
                FState:=psComment;
                Mark;
              end;
      end;

    psComment :
       if Ch in [CR, LF] then
            begin
              FSelEnd:=Length(FSource)-1;
              Found;
              FState:=psNormal;
            end;
  end; { case state }
end;








{ TCppParser - C++ Source parser }

constructor TCppParser.Create;
const
  _Keywords : array[1..107] of string = (
    '__ASM','__CDECL','__CS','__DECLSPEC','__DS','__ES','__EXCEPT','__EXPORT',
    '__FAR','__FASTCALL','__FASTTHIS','__FINALLY','__HUGE','__IMPORT',
    '__INTERRUPT','__LOADDS','__NEAR','__PASCAL','__RTTI','__SAVEREGS',
    '__SEG','__SLOWTHIS','__SS','__TRY','_ASM','_CDECL','_CS','_DS','_ES',
    '_EXPORT','_FAR','_FASTCALL','_HUGE','_IMPORT','_INTERRUPT','_LOADDS',
    '_NEAR','_PASCAL','_SAVEREGS','_SEG','_SS','ASM','AUTO','BOOL','BREAK',
    'CASE','CATCH','CDECL','CHAR','CLASS','CONST','CONST_CAST','CONTINUE',
    'DEFAULT','DELETE','DO','DOUBLE','DYNAMIC_CAST','ELSE','ENUM','EXTERN',
    'FALSE','FAR','FLOAT','FOR','FRIEND','GOTO','HUGE','IF','INLINE','INT',
    'INTERRUPT','LONG','MUTABLE','NAMESPACE','NEAR','NEW','OPERATOR','PASCAL',
    'PRIVATE','PROTECTED','PUBLIC','REGISTER','REINTERPRET_CAST','RETURN',
    'SHORT','SIGNED','SIZEOF','STATIC','STATIC_CAST','STRUCT','SWITCH','TEMPLATE',
    'THIS','THROW','TRUE','TRY','TYPEDEF','TYPEID','UNION','UNSIGNED','USING',
    'VIRTUAL','VOID','VOLATILE','WCHAR_T','WHILE' );
var
  i : Integer;
begin
  inherited Create;
  for i:= Low(_Keywords) to High(_Keywords) do
    FKeywords.Add(_Keywords[i]);
end;

procedure TCppParser.ParseChar(Ch, Next: Char; i: Integer);
begin
  case FState of
    psIdent :
      if not (Ch in IDENT ) then
      begin
        FSelEnd:=Length(FSource)-1;
        if IsKeyWord(FKeyWord) then
        begin
          FState := psKeyword;
          Found;
          { here we should step into the second level of parsing }
        end else
        begin
          { push identifier into list of identifier }
          Found;
        end;
        FState:=psNormal;
      end else
        FKeyword:=FKeyword+Upcase(Ch);

    psNumber :
      if not (Ch in [ZERO..NINE,'.','E','e']) then
      begin
        FSelEnd:=Length(FSource)-1;
        Found;
        FState:=psNormal;                           { push number on stack }
      end;

    psString :
      if Ch = '"' then
         if Next <>'"' then
         begin
           PushChar(Next);
           FSelEnd:=Length(FSource);
           Found;
           FState:=psNormal;                        { push string on stack }
         end;

    psNormal :
      case Ch of
        SPACE,NULL,TAB,CR,LF : ;                    { null characters }

        '>','<','=','[',']','+','-' : ;             { operators }

        ZERO..NINE :                                { numbers }
              begin
                FState:=psNumber;
              end;

        '#','$' :
              if Next in DIGIT  then          { special numbers }
              begin
                FState:=psNumber;
                Mark;
              end;

        'A'..'Z', 'a'..'z', '_' :                   { identifier }
              begin
                FState:=psIdent;
                FKeyword:=Ch;
                Mark;
              end;

        '"' : begin                                 { string }
                FState:=psString;
                Mark;
              end;

        '/' : if Next = '/' then                  { // & /* comment }
              begin
                FState:=psComment;
                FComment:=3;
                Mark;
              end else if Next = '*' then
              begin
                FState:=psComment;
                FComment:=2;
                Mark;
              end;
      end;

    psComment :

      case FComment of
        1 : ;
        2 : if Ch = '*' then
               if Next = '/' then
               begin
                 PushChar(Next);
                 FSelEnd:=Length(FSource);
                 Found;
                 FState:=psNormal;
               end;
        3 : if Ch in [CR, LF] then
            begin
              FSelEnd:=Length(FSource)-1;
              Found;
              FState:=psNormal;
            end;
      end;
  end; { case state }
end;




///////////////////////////////////////////
///////// ParserCallback Class ////////////
///////////////////////////////////////////


type
  TParserCallBack = class
    class procedure FoundColor(Sender: TObject);
    class procedure FoundMono(Sender: TObject);
    class procedure Replace(Sender: TObject);
  end;

class procedure TParserCallBack.FoundColor(Sender: TObject);
var
  aParser : TParser;
begin
  aParser := Sender as TParser;
  case aParser.State of
    psKeyword : aParser.HighLight('{\cf1\b ','}');   { white bold }
    psComment : aParser.HighLight('{\cf3\i ','}');   { gray  italic}
  end;
end;

class procedure TParserCallBack.FoundMono(Sender: TObject);
var
  aParser : TParser;
begin
  aParser := Sender as TParser;
  case aParser.State of
    psKeyword : aParser.HighLight('{\b ','}');   { bold }
    psComment : aParser.HighLight('{\i ','}');   { italic}
  end;
end;

class procedure TParserCallBack.Replace(Sender: TObject);
var
  aParser : TParser;
  Ch : Char;
  S: String;
begin
  aParser := Sender as TParser;
  S := aParser.Source;
  Ch := S[Length(S)];
  case Ch of
    '{' : Insert('\',S,Length(S));
    '}' : Insert('\',S,Length(S));
    LF  : Insert('\par ',S,Length(S));
    else exit;
  end;
  aParser.Source:=S;
end;

{ converts a Delphi TColor into a RTF-color table string }
function ColorToRtf(aColor:TColor): String;
begin
  aColor:=ColorToRGB(aColor);
  Result:='\red'+IntToStr(GetRValue(aColor))+
          '\green'+IntToStr(GetGValue(aColor))+
          '\blue'+IntToStr(GetBValue(aColor))+';';
end;



function SourceToRtf(const aFile: String; aMono,aHeader: Boolean): String;
var
  aParser   : TParser;
  aExt      : String[10];
  RtfHeader : String;
begin
  RtfHeader :=
    '{\rtf1\ansi\deff0\deftab720'
   +'{\fonttbl'
     +'{\f0\fmodern Courier New;}}'
   +'{\colortbl'+
      ColorToRtf(clBlack)+
      ColorToRtf(clWhite);

   if not aMono then                { add yellow and silver }
      RtfHeader := RtfHeader +
        ColorToRtf(clYellow)+
        ColorToRtf(clSilver);

  RtfHeader:= RtfHeader +'}' +'\deflang1031\pard\plain\f0\fs20';

  if not aMono then RtfHeader:= RtfHeader + '\cf2';   { yellow as default }

  aExt:=LowerCase(ExtractFileExt(aFile));
  if Pos(aExt,'*.pas;*.dpr;*.int;*.inc') > 0 then
     aParser := TPascalParser.Create
  else if Pos(aExt,'*.asm') > 0 then
     aParser := TAsmParser.Create
  else if Pos(aExt,'*.c;*.cpp;*.h;*.hpp') > 0 then
     aParser := TCppParser.Create
  else
     raise Exception.Create('Unknown source yet !');

  Result:= strFileLoad(aFile);

  try
    with TParserCallBack do begin
      if aMono then
         aParser.OnFound:= FoundMono
      else
         aParser.OnFound:= FoundColor;
      aParser.OnReplace := Replace;
    end;

    Result := aParser.Parse(Result);
    if aHeader then
       Result := RtfHeader+Result+'}';
  finally
    aParser.Free;
  end;
end;


function strFileLoad(const aFile: String): String;
var
  aStr : TStrings;
begin
  Result:='';
  aStr:=TStringList.Create;
  try
    aStr.LoadFromFile(aFile);
    Result:=aStr.Text;
  finally
    aStr.Free;
  end;
end;



end.

